<?php
declare(strict_types=1);

require_once __DIR__ . '/../models/Comision.php';
require_once __DIR__ . '/../models/User.php';
require_once __DIR__ . '/../models/ComisionConstancia.php';

class ComisionController extends Controller {

    /* =========================================================
     * Helpers de paths/urls (funcionan con o sin /public)
     * ========================================================= */
    private function pubRootFs(): string {
        return realpath(__DIR__ . '/../../public') ?: (__DIR__ . '/../../public');
    }
    private function baseUrl(): string {
        return rtrim((string)(defined('BASE_URL') ? BASE_URL : ''), '/');
    }
    private function publicBaseRoot(): string { return $this->baseUrl(); }
    private function buildPublicUrl(string $publicRelative): string {
        $rel = '/' . ltrim($publicRelative, '/');
        return $this->baseUrl() . $rel;
    }
    private function resolvePublicUrl(string $publicUrl): array {
        $base = $this->baseUrl(); $rel  = $publicUrl;
        if (stripos($publicUrl, 'http') === 0) {
            if (stripos($publicUrl, $base) === 0) { $rel = substr($publicUrl, strlen($base)); }
            else { return [null, basename(parse_url($publicUrl, PHP_URL_PATH) ?: 'archivo'), '']; }
        }
        $rel = '/' . ltrim($rel, '/');
        $abs  = realpath($this->pubRootFs() . $rel);
        $name = basename(parse_url($rel, PHP_URL_PATH) ?: 'archivo');
        $root = realpath($this->pubRootFs());
        if (!$abs || !$root || strpos($abs, $root) !== 0) return [null, $name, ''];
        return [$abs, $name, ''];
    }
    private function detectMime(string $absPath): string {
        $ext = strtolower(pathinfo($absPath, PATHINFO_EXTENSION));
        $map = [
            'pdf'=>'application/pdf','png'=>'image/png','jpg'=>'image/jpeg','jpeg'=>'image/jpeg',
            'webp'=>'image/webp','gif'=>'image/gif','doc'=>'application/msword',
            'docx'=>'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
            'xls'=>'application/vnd.ms-excel','xlsx'=>'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
            'ppt'=>'application/vnd.ms-powerpoint','pptx'=>'application/vnd.openxmlformats-officedocument.presentationml.presentation',
        ];
        if (isset($map[$ext])) return $map[$ext];
        $fi = new finfo(FILEINFO_MIME_TYPE);
        return $fi->file($absPath) ?: 'application/octet-stream';
    }
    private function ensureDir(string $dir): bool {
        return is_dir($dir) || @mkdir($dir, 0777, true) || is_dir($dir);
    }
    /** Stream inline con soporte Range (visor PDF) */
    private function streamInline(string $abs, string $mime, string $downloadName): void {
        $size = filesize($abs); $start=0; $end=$size-1; $status=200;
        if (isset($_SERVER['HTTP_RANGE']) && preg_match('/bytes=(\d+)-(\d*)/i', $_SERVER['HTTP_RANGE'], $m)) {
            $start=(int)$m[1]; if ($m[2] !== '') $end=(int)$m[2]; if ($end >= $size) $end=$size-1;
            if ($start > $end) { http_response_code(416); exit; }
            $status=206; header("Content-Range: bytes $start-$end/$size");
        }
        if ($status===206) header('HTTP/1.1 206 Partial Content');
        header('Content-Type: '.$mime);
        header('Content-Disposition: inline; filename="'.rawurlencode($downloadName).'"');
        header('Accept-Ranges: bytes');
        header('Content-Length: '.(($end-$start)+1));
        header('Cache-Control: private, max-age=300');
        header("X-Frame-Options: SAMEORIGIN");
        header("Content-Security-Policy: frame-ancestors 'self'");
        $fp=fopen($abs,'rb'); if(!$fp){ http_response_code(404); exit; }
        if ($start>0) fseek($fp,$start);
        $left = ($end-$start)+1; $buf=8192;
        while(!feof($fp) && $left>0){
            $chunk=fread($fp, min($buf,$left)); if($chunk===false) break;
            echo $chunk; flush(); $left -= strlen($chunk);
        }
        fclose($fp); exit;
    }

    /* ======================= Helpers de negocio ======================= */

    /** Participantes: chips + del sistema + chofer (sin duplicados) */
    private function buildParticipantsList(string $participantesText, array $partSis, string $choferNombre = ''): array {
        $list = [];
        $chips = array_filter(array_map('trim', explode(',', (string)$participantesText)));
        foreach ($chips as $c) if ($c !== '') $list[] = $c;
        foreach ($partSis as $ps) {
            $name = trim($ps['nombre'] ?? ''); if ($name !== '') $list[] = $name;
        }
        $ch = trim($choferNombre); if ($ch !== '') $list[] = $ch;
        $seen = []; $final = [];
        foreach ($list as $n) { $k = mb_strtolower($n); if (isset($seen[$k])) continue; $seen[$k]=true; $final[]=$n; }
        return $final;
    }

    /** Estimación de combustible según vehículo/KM (galón/litro) */
    private function calcularCombustible(?string $vehiculoNombre, ?float $kmTotales): array {
        $vehiculo = strtoupper(trim((string)$vehiculoNombre));
        $tabla = [
            'TOYOTA FORTUNER'      => ['tipo'=>'GASOLINA','km_gal'=>22.0],
            'TOYOTA HILUX'         => ['tipo'=>'GASOLINA','km_gal'=>36.0],
            'CHEVROLET D-MAX'      => ['tipo'=>'DIÉSEL',  'km_gal'=>38.0],
            'VITARA 3 PUERTAS'     => ['tipo'=>'GASOLINA','km_gal'=>50.0],
        ];
        $kmPorGal = $tabla[$vehiculo]['km_gal'] ?? 30.0;
        $tipo     = $tabla[$vehiculo]['tipo']   ?? 'GASOLINA';
        $km = round(max(0.0, (float)$kmTotales), 2);

        $precioGas    = defined('FUEL_GAS_PRICE_USD')    ? (float)FUEL_GAS_PRICE_USD    : 2.88;
        $precioDiesel = defined('FUEL_DIESEL_PRICE_USD') ? (float)FUEL_DIESEL_PRICE_USD : 2.80;
        $precioGal = ($tipo === 'DIÉSEL') ? $precioDiesel : $precioGas;

        $galones = $km > 0 ? ($km / $kmPorGal) : 0.0;
        $LITROS_POR_GALON = 3.785411784;
        $costo  = $galones * $precioGal;

        return [
            'vehiculo'=>$vehiculoNombre ?: '', 'tipo'=>$tipo,
            'km_totales'=>$km, 'km_por_galon'=>$kmPorGal, 'galones'=>$galones,
            'precio_galon'=>$precioGal, 'costo_usd'=>$costo,
            'km_por_litro'=>($kmPorGal / $LITROS_POR_GALON),
            'litros'=>($galones * $LITROS_POR_GALON),
            'precio_litro'=>($precioGal / $LITROS_POR_GALON),
        ];
    }

    private function armarMovilidad(array $item): array {
        $origen  = trim((string)($item['origen_ciudad'] ?? ''));
        $ida     = trim((string)($item['destino_ida_ciudad'] ?? ''));
        $retorno = trim((string)($item['destino_retorno_ciudad'] ?? ''));
        $km      = isset($item['total_km']) ? (float)$item['total_km'] : 0.0;
        $veh     = (string)($item['vehiculo_nombre'] ?? '');
        $comb    = $this->calcularCombustible($veh, $km);
        return [
            'origen'=>$origen,'destino'=>$ida,'retorno'=>$retorno,
            'km_totales'=>$km,'vehiculo'=>$veh,
            'chofer'=>(string)($item['chofer_nombre'] ?? ''),'combustible'=>$comb
        ];
    }

    /** Viáticos (monto por persona) */
    private function calcularViaticos(string $fini, string $ffin, array $nombres): array {
        try { $d1 = new DateTime($fini); $d2 = new DateTime($ffin); $dias = $d1->diff($d2)->days + 1; }
        catch (\Throwable $e) { $dias = 1; }
        $tarifa = ($dias <= 1) ? 40 : 80;
        $detalles = []; $total = 0.0;
        foreach ($nombres as $nom) {
            $m = ($dias <= 1) ? 40.0 : (80.0 * $dias);
            $detalles[] = ['nombre'=>$nom,'dias'=>$dias,'monto'=>$m];
            $total += $m;
        }
        return ['dias'=>$dias,'tarifa'=>$tarifa,'detalles'=>$detalles,'total_global'=>$total,
                'formula'=>($dias<=1)?'1 día × USD 40':($dias.' días × USD 80')];
    }

    /** Viáticos por persona + FONDO solo al chofer (para estimados) */
    private function viaticosPorPersonaConFondo(
        string $fini, string $ffin, array $nombres, string $choferNombre, float $combustibleUSD
    ): array {
        try { $d1=new DateTime($fini); $d2=new DateTime($ffin); $dias=$d1->diff($d2)->days + 1; } catch (\Throwable $e){ $dias=1; }
        $montoViatico = ($dias <= 1) ? 40.0 : (80.0 * $dias);
        $choferKey = mb_strtolower(trim($choferNombre));
        $out=[]; $totalGlobal=0.0; $seen=[];
        foreach ($nombres as $nom) {
            $k=mb_strtolower(trim($nom)); if($k==='') continue; if(isset($seen[$k])) continue; $seen[$k]=1;
            $via=$montoViatico;
            $fondo = ($k === $choferKey) ? max(0.0,(float)$combustibleUSD) : 0.0;
            $sum = $via + $fondo;
            $out[] = ['nombre'=>$nom,'dias'=>$dias,'viaticos_usd'=>$via,'fondo_comb_usd'=>$fondo,'total_persona'=>$sum];
            $totalGlobal += $sum;
        }
        return ['dias'=>$dias,'detalles'=>$out,'total_global'=>$totalGlobal,'formula'=>($dias<=1)?'1 día × USD 40':($dias.' días × USD 80')];
    }

    /* ============================ Acciones ============================ */

    /** Listado: autoridades ven todo; otros, solo lo suyo */
    public function index(): void { require_login();
        $u = user(); $m = new Comision();
        $isAdminOrAlcalde = in_array(strtoupper($u['rol']), ['ADMIN','ALCALDE','PRESIDENTE'], true);
        $items = $isAdminOrAlcalde ? $m->allWithConst() : $m->allByCreadorWithConst((int)$u['id']);
        $this->view('comision/index', ['items'=>$items, 'u'=>$u]);
    }

    /** Detalle */
    public function show(int $id): void { require_login();
        $u=user(); $m=new Comision(); $item=$m->find($id);
        if(!$item){ http_response_code(404); exit('Comisión no encontrada'); }
        $rol=strtoupper($u['rol'] ?? ''); $esCreador=((int)$item['creador_id']===(int)$u['id']);
        if(!$esCreador && !in_array($rol,['ALCALDE','ADMIN','PRESIDENTE'],true)){ http_response_code(403); exit('No autorizado'); }

        // Participantes del sistema
        $participantesSistema=[]; try{
            $db=DB::conn();
            $stmt=$db->prepare("SELECT u.nombre, r.nombre AS rol
                                  FROM comision_participantes cp
                                  JOIN users u ON u.id=cp.usuario_id
                                  LEFT JOIN roles r ON r.id=u.rol_id
                                 WHERE cp.comision_id=?");
            $stmt->execute([$id]); $participantesSistema=$stmt->fetchAll();
        }catch(\Throwable $e){}

        $nombres  = $this->buildParticipantsList((string)($item['participantes_text'] ?? ''), $participantesSistema, (string)($item['chofer_nombre'] ?? ''));
        $viaticos = $this->calcularViaticos((string)$item['fecha_inicio'], (string)$item['fecha_fin'], $nombres);
        $movilidad= $this->armarMovilidad($item);

        $this->view('comision/show', compact('item','viaticos','movilidad'));
    }

    /** Crear: calcula estimados y asigna fondo al chofer */
    public function create(): void { require_login();
        $u=user(); $areas=(new Area())->all(); $users=(new User())->all(); $deps=[]; $error=null;

        if (is_post()) {
            $participantes_text = trim($_POST['participantes_text'] ?? '');
            $chofer = trim($_POST['chofer_nombre'] ?? '');
            $vehic  = trim($_POST['vehiculo_nombre'] ?? '');
            $km     = isset($_POST['total_km']) ? (float)$_POST['total_km'] : 0.0;

            $data = [
                'titulo'                 => trim($_POST['titulo'] ?? ''),
                'descripcion'            => trim($_POST['descripcion'] ?? ''),
                'fecha_inicio'           => $_POST['fecha_inicio'] ?? null,
                'fecha_fin'              => $_POST['fecha_fin'] ?? null,
                'area_id'                => (int)($_POST['area_id'] ?? 0),
                'department_id'          => (int)($_POST['department_id'] ?? 0),
                'participantes'          => array_map('intval', $_POST['participantes'] ?? []),
                'participantes_text'     => $participantes_text,
                'chofer_nombre'          => $chofer,
                'vehiculo_nombre'        => $vehic,
                'origen_ciudad'          => trim($_POST['origen_ciudad'] ?? ''),
                'destino_ida_ciudad'     => trim($_POST['destino_ida_ciudad'] ?? ''),
                'destino_retorno_ciudad' => trim($_POST['destino_retorno_ciudad'] ?? ''),
                'total_km'               => $km,
                'creador_id'             => $u['id'],
            ];

            if ($data['titulo']==='' || !$data['fecha_inicio'] || !$data['fecha_fin']) {
                $error="Completa Título y rango de fechas.";
                $this->view('comision/create', compact('areas','users','deps','error'));
                return;
            }

            // ====== Estimados ======
            $comb = $this->calcularCombustible($vehic, $km);
            $combustibleUSD = (float)($comb['costo_usd'] ?? 0.0);

            $chips = array_filter(array_map('trim', explode(',', $participantes_text)));
            if ($chofer!=='') $chips[] = $chofer;

            $det = $this->viaticosPorPersonaConFondo(
                (string)$data['fecha_inicio'], (string)$data['fecha_fin'],
                $chips, $chofer, $combustibleUSD
            );

            $dias = (int)($det['detalles'][0]['dias'] ?? 1);
            $viaticosTotal = 0.0; $choferTotal = 0.0;
            foreach ($det['detalles'] as $p) {
                $viaticosTotal += (float)$p['viaticos_usd'];
                if (mb_strtolower(trim($p['nombre'])) === mb_strtolower(trim($chofer))) {
                    $choferTotal = (float)$p['total_persona']; // viático + fondo
                }
            }
            $totalGeneral = $viaticosTotal + $combustibleUSD;

            // Guardar campos estimados
            $data['dias_viatico']         = $dias;
            $data['viaticos_total_est']   = $viaticosTotal;
            $data['combustible_usd_est']  = $combustibleUSD;
            $data['chofer_total_est']     = $choferTotal;
            $data['total_general_est']    = $totalGeneral;

            $id = (new Comision())->create($data);

            // ===== Adjuntar plan =====
            if (!empty($_FILES['planificacion']['name'])) {
                $file = $_FILES['planificacion']; $ok = $file['error'] === UPLOAD_ERR_OK;
                if ($file['error'] === UPLOAD_ERR_INI_SIZE || $file['error'] === UPLOAD_ERR_FORM_SIZE) {
                    $ok=false; (new Trazabilidad())->log('COMISIONES','ERR_FILE_SIZE',$u['id'],$id,(string)$file['error']);
                }
                $mime=''; if($ok){ $f=new finfo(FILEINFO_MIME_TYPE); $det=$f->file($file['tmp_name']); $mime=$det ?: ($file['type'] ?? ''); }
                $permitidos=['application/pdf','application/vnd.openxmlformats-officedocument.wordprocessingml.document',
                             'application/msword','application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
                             'image/jpeg','image/png'];
                if ($ok && !in_array($mime,$permitidos,true)){ $ok=false; (new Trazabilidad())->log('COMISIONES','ERR_MIME_PLAN',$u['id'],$id,$mime ?: 'sin-mime'); }
                $dirFs = $this->pubRootFs() . '/archivos/comisiones/' . $id;
                if ($ok && !$this->ensureDir($dirFs)){ $ok=false; (new Trazabilidad())->log('COMISIONES','ERR_DIR_PLAN',$u['id'],$id,$dirFs); }
                if ($ok) {
                    $orig=$file['name']; $ext=pathinfo($orig,PATHINFO_EXTENSION);
                    $safe=preg_replace('/[^A-Za-z0-9_\-.]/','_', pathinfo($orig,PATHINFO_FILENAME));
                    $fname=$safe.'_'.date('Ymd_His').($ext?'.'.$ext:''); $dest=$dirFs.'/'.$fname;
                    if (move_uploaded_file($file['tmp_name'],$dest)) {
                        $publicUrl=$this->buildPublicUrl('/archivos/comisiones/'.$id.'/'.$fname);
                        (new Comision())->anexarPlan($id, ['nombre'=>$orig,'ruta'=>$publicUrl,'mime'=>$mime,'tamano'=>(int)($file['size'] ?? 0)]);
                        (new Trazabilidad())->log('COMISIONES','ADJUNTO_PLAN',$u['id'],$id,$orig);
                    } else {
                        (new Trazabilidad())->log('COMISIONES','ERR_MOVE_PLAN',$u['id'],$id,$orig ?? 'sin-nombre');
                    }
                }
            }

            (new Trazabilidad())->log('COMISIONES','CREAR',$u['id'],$id,
                sprintf('dias=%d; via=%.2f; comb=%.2f; chofer=%.2f; total=%.2f',
                        $dias,$viaticosTotal,$combustibleUSD,$choferTotal,$totalGeneral)
            );

            redirect('/comision/show/' . $id);
        }

        $this->view('comision/create', compact('areas','users','deps','error'));
    }

    /** Ver constancia */
    public function constancia(int $comisionId): void { require_login();
        $u = user();

        $com = (new Comision())->find($comisionId);
        if (!$com) { http_response_code(404); exit('Comisión no encontrada'); }

        $rolUp = strtoupper($u['rol'] ?? '');
        $esAutoridad = in_array($rolUp, ['PRESIDENTE','ALCALDE','ADMIN'], true);
        $esCreador   = (int)$com['creador_id'] === (int)$u['id'];
        if (!$esAutoridad && !$esCreador) { http_response_code(403); exit('No autorizado'); }

        $mConst = new ComisionConstancia();
        $const = $mConst->findByComision($comisionId);
        if (!$const) { http_response_code(404); exit('No hay constancia generada'); }

        if ($esAutoridad && is_post() && !empty($_FILES['pdf_firmado']['name'])) {
            $f = $_FILES['pdf_firmado'];
            $ok = $f['error'] === UPLOAD_ERR_OK && (strtolower(pathinfo($f['name'], PATHINFO_EXTENSION)) === 'pdf');

            if ($ok) {
                $dirFs = $this->pubRootFs() . '/constancias/comisiones/' . $comisionId;
                $this->ensureDir($dirFs);
                $fname = 'constancia_firmada_' . date('Ymd_His') . '.pdf';
                $dest  = $dirFs . '/' . $fname;

                if (move_uploaded_file($f['tmp_name'], $dest)) {
                    $public = $this->buildPublicUrl('/constancias/comisiones/' . $comisionId . '/' . $fname);
                    $mConst->setFirmado((int)$const['id'], $public);
                    (new Trazabilidad())->log('COMISIONES','CONSTANCIA_FIRMADA', $u['id'], $comisionId, $const['codigo']);
                }
            }
        }

        $this->view('comision/constancia', [
            'const'      => $const,
            'comisionId' => $comisionId,
            'esAlcalde'  => $esAutoridad,
            'esCreador'  => $esCreador
        ]);
    }

    /** Aprobar/Rechazar + generar constancia con QR */
    public function aprobar(int $id): void { require_login();
        $u = user();
        $rolUp = strtoupper($u['rol'] ?? '');
        if (!in_array($rolUp, ['PRESIDENTE','ALCALDE','ADMIN'], true)) { http_response_code(403); exit('No autorizado'); }

        $mCom = new Comision();
        $item = $mCom->find($id);
        if (!$item) { http_response_code(404); exit('No encontrada'); }

        $participantesSistema = [];
        try {
            $db = DB::conn();
            $stmt = $db->prepare("SELECT u.nombre, r.nombre AS rol
                                    FROM comision_participantes cp
                                    JOIN users u ON u.id=cp.usuario_id
                                    LEFT JOIN roles r ON r.id=u.rol_id
                                   WHERE cp.comision_id=?");
            $stmt->execute([$id]);
            $participantesSistema = $stmt->fetchAll();
        } catch (\Throwable $e) {}

        $nombres  = $this->buildParticipantsList((string)($item['participantes_text'] ?? ''), $participantesSistema, (string)($item['chofer_nombre'] ?? ''));
        $viaticos = $this->calcularViaticos((string)$item['fecha_inicio'], (string)$item['fecha_fin'], $nombres);
        $movilidad= $this->armarMovilidad($item);

        if (is_post()) {
            $estado = ($_POST['estado'] ?? '') === 'APROBADA' ? 'APROBADA' : 'RECHAZADA';
            $coment = trim($_POST['comentario'] ?? '');

            $mCom->updateEstado($id, $estado, $coment, $u['id']);
            (new Trazabilidad())->log('COMISIONES', $estado, $u['id'], $id, $coment);

            if ($estado === 'APROBADA') {
                $this->generarConstanciaAprobacion($item, $participantesSistema, $coment, $u);
            }
            redirect('/comision/index'); return;
        }

        $this->view('comision/aprobar', compact('item','participantesSistema','viaticos','movilidad'));
    }

    /* =============== Verificación pública y previews =============== */

    public function verificar_constancia(string $codigo): void {
        $mConst = new ComisionConstancia();
        $const  = $mConst->findByCodigo($codigo);
        if (!$const) {
            http_response_code(404);
            echo "<!doctype html><meta charset='utf-8'><title>No encontrada</title>
                  <style>body{font-family:system-ui,-apple-system,Segoe UI,Roboto,Helvetica,Arial; padding:20px; background:#f3f4f6}</style>
                  <div style='max-width:720px;margin:auto;background:#fff;border:1px solid #e5e7eb;border-radius:12px;padding:16px'>
                    <h2>Constancia no encontrada</h2>
                    <p>El código <b>".htmlspecialchars($codigo)."</b> no existe o fue anulado.</p>
                  </div>";
            exit;
        }
        $estado = strtoupper($const['estado'] ?? 'GENERADA');
        $valida = ($estado === 'FIRMADA' || $estado === 'GENERADA');
        $ok = $valida ? '✅ VÁLIDA' : '❌ NO VÁLIDA';
        $pdf = htmlspecialchars((string)($const['pdf_path'] ?? $const['pdf_url'] ?? ''));
        $firmado = htmlspecialchars((string)($const['firmado_path'] ?? $const['firmado_url'] ?? ''));

        echo "<!doctype html><meta charset='utf-8'><title>Verificación {$codigo}</title>
        <div style='max-width:820px;margin:auto;background:#fff;border:1px solid #e5e7eb;border-radius:12px;padding:16px;font-family:system-ui,-apple-system,Segoe UI,Roboto,Helvetica,Arial'>
          <h2>Verificación de constancia</h2>
          <p><b>Código:</b> ".htmlspecialchars($const['codigo'])."</p>
          <p><b>Estado:</b> {$ok} <span style='color:#6b7280'>(actual: ".htmlspecialchars($estado).")</span></p>
          <p><b>Generada el:</b> ".htmlspecialchars($const['created_at'] ?? '')."</p>
          <p><b>Documento:</b> ".
             ($pdf ? "<a style='display:inline-block;padding:8px 12px;border:1px solid #cbd5e1;border-radius:10px;text-decoration:none' target='_blank' href='{$pdf}'>Abrir constancia</a>" : "<span style='color:#6b7280'>No disponible</span>")
             ." ".($firmado ? "<a style='display:inline-block;padding:8px 12px;border:1px solid #cbd5e1;border-radius:10px;text-decoration:none' target='_blank' href='{$firmado}'>PDF firmado</a>" : "")."
          </p>
          <p style='color:#6b7280'>Portal de validación del Sistema de Gestión de Comisiones del <b>COMAGA</b>.</p>
        </div>";
        exit;
    }

    public function preview_plan(int $comisionId): void { require_login();
        $u = user();
        $com = (new Comision())->find($comisionId);
        if (!$com) { http_response_code(404); exit('Comisión no encontrada'); }
        $rol = strtoupper($u['rol'] ?? '');
        $esCreador = (int)$com['creador_id'] === (int)$u['id'];
        if (!$esCreador && !in_array($rol, ['ALCALDE','ADMIN','PRESIDENTE'], true)) { http_response_code(403); exit('No autorizado'); }

        $publicUrl = (string)($com['plan_ruta'] ?? '');
        if ($publicUrl === '') { http_response_code(404); exit('Sin archivo'); }

        [$abs,$name,$mime] = $this->resolvePublicUrl($publicUrl);
        if (!$abs || !is_file($abs)) { http_response_code(404); exit('Archivo no encontrado'); }
        if ($mime === '') $mime = $this->detectMime($abs);
        $this->streamInline($abs, $mime, $name);
    }

    public function preview_firmado(int $comisionId): void { require_login();
        $u = user();
        $com = (new Comision())->find($comisionId);
        if (!$com) { http_response_code(404); exit('Comisión no encontrada'); }

        $rol = strtoupper($u['rol'] ?? '');
        $esCreador = (int)$com['creador_id'] === (int)$u['id'];
        if (!$esCreador && !in_array($rol, ['ALCALDE','PRESIDENTE'], true)) { http_response_code(403); exit('No autorizado'); }

        $const = (new ComisionConstancia())->findByComision($comisionId);
        if (!$const) { http_response_code(404); exit('No hay constancia'); }

        $publicUrl = (string)($const['firmado_path'] ?? $const['firmado_url'] ?? '');
        if ($publicUrl === '') { http_response_code(404); exit('No hay PDF firmado'); }

        [$abs,$name,$mime] = $this->resolvePublicUrl($publicUrl);
        if (!$abs || !is_file($abs)) { http_response_code(404); exit('Archivo no encontrado'); }
        if ($mime === '') $mime = $this->detectMime($abs);
        $this->streamInline($abs, $mime, $name);
    }

    /** Genera constancia con QR (URL de verificación) y guarda PDF/HTML bajo /public/constancias/comisiones/{id} */
    private function generarConstanciaAprobacion(array $comision, array $partSis, string $comentarioAlcalde, array $alcalde): void {
        $comId  = (int)$comision['id'];
        $dirFs  = $this->pubRootFs() . '/constancias/comisiones/' . $comId;
        if (!$this->ensureDir($dirFs)) {
            (new Trazabilidad())->log('COMISIONES','CONSTANCIA_ERR_DIR', $alcalde['id'], $comId, $dirFs);
            return;
        }

        // obtener/crear registro de constancia para generar código final estable
        $mConst = new ComisionConstancia();
        $exist  = $mConst->findByComision($comId);
        if (!$exist) {
            $tmpCode = sprintf('TMP-%06d-%s', $comId, date('YmdHis'));
            $idRow   = $mConst->create($comId, $tmpCode, '', '', null);
            $codigoFinal = sprintf('CONST-COM-%06d-%s', (int)$idRow, date('Y'));
            $db = DB::conn();
            $upd = $db->prepare("UPDATE comision_constancias SET codigo=? WHERE id=?");
            $upd->execute([$codigoFinal, $idRow]);
            $const = $mConst->find($idRow);
        } else {
            $const = $exist;
        }
        $codigo = (string)$const['codigo'];

        // URL pública de verificación
        $verifyUrl = $this->publicBaseRoot() . '/constancia/verificar/' . rawurlencode($codigo);
        $qrText    = $verifyUrl; // El QR apunta a la verificación pública

        // QR simple (servicio externo)
        $qrPng = $dirFs . '/qr.png';
        $qrOk  = false;
        try {
            $png = @file_get_contents('https://api.qrserver.com/v1/create-qr-code/?size=160x160&data=' . urlencode($qrText));
            if ($png) { file_put_contents($qrPng, $png); $qrOk = true; }
        } catch (\Throwable $e) {}
        $qrHtml = $qrOk
            ? '<img src="'.$this->buildPublicUrl('/constancias/comisiones/'.$comId.'/qr.png').'" width="160" height="160" alt="QR de verificación" />'
            : '<div style="width:160px;height:160px;border:1px dashed #999;display:flex;align-items:center;justify-content:center;font-size:12px;">QR</div>';

        // Participantes chips
        $pt = trim((string)($comision['participantes_text'] ?? ''));
        $chipsText = array_filter(array_map('trim', explode(',', $pt)));
        $chipsHtml = '';
        foreach ($chipsText as $c)  { if ($c!=='') $chipsHtml .= '<span style="display:inline-block;background:#eef2ff;border-radius:999px;padding:6px 10px;margin:3px;border:1px solid #e5e7eb;">'.htmlspecialchars($c).'</span> '; }
        foreach ($partSis as $ps){
            $n = trim((string)($ps['nombre'] ?? ''));
            $r = trim((string)($ps['rol'] ?? ''));
            if ($n==='') continue;
            $chipsHtml .= '<span style="display:inline-block;background:#e7fbea;border-radius:999px;padding:6px 10px;margin:3px;border:1px solid #e5e7eb;">'.htmlspecialchars($n).($r ? ' ('.htmlspecialchars($r).')' : '').'</span> ';
        }

        // Movilidad / combustible
        $mov  = $this->armarMovilidad($comision);
        $comb = $mov['combustible'];
        $kmTxt = number_format((float)$mov['km_totales'],2);
        $combCosto = (float)($comb['costo_usd'] ?? ($comb['costo'] ?? 0));

        // Viáticos
        $nombres  = $this->buildParticipantsList($pt, $partSis, (string)($comision['chofer_nombre'] ?? ''));
        $viaticos = $this->calcularViaticos((string)$comision['fecha_inicio'], (string)$comision['fecha_fin'], $nombres);
        $granTotal = $viaticos['total_global'] + $combCosto;

        // HTML plantilla — actualizado (sin Área/Depto y con firma única)
        $html = '
        <html><head><meta charset="utf-8"><style>
          @page { margin: 24px 24px 28px 24px; }
          body{font-family: DejaVu Sans, sans-serif; font-size:12px; color:#1f2937;}
          .wrap{ max-width:900px; margin:0 auto; }
          .hdr{ display:flex; align-items:center; gap:12px; margin-bottom:8px; }
          .logo{ width:46px; height:46px; border-radius:10px; border:1px solid #cbd5e1; overflow:hidden; }
          .title{ font-size:18px; font-weight:700; color:#1e3a8a; }
          .muted{ color:#6b7280; }
          .grid{ display:grid; grid-template-columns: 1fr 1fr; gap:12px; }
          .box{ border:1px solid #e5e7eb; border-radius:10px; padding:10px; }
          .meta td{ padding:3px 6px; vertical-align:top; }
          .tbl{ width:100%; border-collapse:collapse; margin-top:6px; }
          .tbl th,.tbl td{ border:1px solid #e5e7eb; padding:6px; }
          .tbl th{ background:#f8fafc; text-align:left; }
          .kpis{ display:grid; grid-template-columns:1fr 1fr 1fr; gap:8px; margin-top:6px; }
          .kpi{ border:1px dashed #cbd5e1; border-radius:10px; padding:8px; }
          .kpi b{ font-size:12px; }
          .total{ background:#eff6ff; border:1px solid #bfdbfe; }
          .qr{ text-align:center; }
          .sign{ display:flex; gap:16px; margin-top:16px; }
          .sig{ flex:1; text-align:center; }
          .sig .line{ border-top:1px solid #111827; margin-top:48px; padding-top:4px; }
          @media (max-width:680px){
            .grid{ grid-template-columns:1fr; }
            .kpis{ grid-template-columns:1fr; }
          }
        </style></head><body><div class="wrap">
          <div class="hdr">
            <div class="logo"><img src="'.$this->buildPublicUrl('/img/logoc.png').'" alt="Logo" style="width:100%;height:100%;object-fit:cover"/></div>
            <div>
              <div class="title">CONSTANCIA DE APROBACIÓN DE COMISIÓN</div>
              <div class="muted">Sistema de Gestión de Comisiones · COMAGA</div>
            </div>
          </div>

          <table class="meta">
            <tr><td><b>Código:</b></td><td>'.htmlspecialchars($codigo).'</td></tr>
            <tr><td><b>Fecha de emisión:</b></td><td>'.date('Y-m-d H:i').'</td></tr>
            <tr><td><b>Título:</b></td><td>'.htmlspecialchars((string)$comision['titulo']).'</td></tr>
            <tr><td><b>Rango:</b></td><td>'.htmlspecialchars((string)$comision['fecha_inicio']).' → '.htmlspecialchars((string)$comision['fecha_fin']).'</td></tr>
            <tr><td><b>Presidente COMGA:</b></td><td>'.htmlspecialchars((string)$alcalde['nombre']).'</td></tr>
          </table>

          <div class="box" style="margin-top:8px;">
            <b>Participantes:</b><br>'.$chipsHtml.'
          </div>

          <div class="grid" style="margin-top:8px;">
            <div class="box">
              <b>Traslado</b>
              <table class="meta" style="margin-top:6px">
                <tr><td><b>Origen:</b></td><td>'.htmlspecialchars((string)$mov['origen'] ?: '—').'</td></tr>
                <tr><td><b>Destino (ida):</b></td><td>'.htmlspecialchars((string)$mov['destino'] ?: '—').'</td></tr>
                <tr><td><b>Destino (retorno):</b></td><td>'.htmlspecialchars((string)$mov['retorno'] ?: '—').'</td></tr>
                <tr><td><b>Vehículo:</b></td><td>'.htmlspecialchars((string)$mov['vehiculo'] ?: '—').'</td></tr>
                <tr><td><b>Chofer:</b></td><td>'.htmlspecialchars((string)$mov['chofer'] ?: '—').'</td></tr>
                <tr><td><b>Kilómetros:</b></td><td>'.$kmTxt.' km</td></tr>
              </table>
            </div>
            <div class="box">
              <b>Combustible</b>
              <table class="meta" style="margin-top:6px">
                <tr><td><b>Tipo:</b></td><td>'.htmlspecialchars((string)($comb['tipo'] ?? '—')).'</td></tr>
                '.(isset($comb['km_por_galon']) ? '
                  <tr><td><b>Rendimiento:</b></td><td>~ '.number_format((float)$comb['km_por_galon'],1).' km/gal</td></tr>
                  <tr><td><b>Consumo:</b></td><td>'.number_format((float)$comb['galones'],2).' gal</td></tr>
                  <tr><td><b>Precio/galón:</b></td><td>$'.number_format((float)$comb['precio_galon'],2).'</td></tr>
                ' : '
                  <tr><td><b>Rendimiento:</b></td><td>~ '.number_format((float)($comb['km_por_litro'] ?? 0),1).' km/L</td></tr>
                  <tr><td><b>Consumo:</b></td><td>'.number_format((float)($comb['litros'] ?? 0),2).' L</td></tr>
                  <tr><td><b>Precio/litro:</b></td><td>$'.number_format((float)($comb['precio_litro'] ?? 0),2).'</td></tr>
                ').'
                <tr><td><b>Costo:</b></td><td><b>$'.number_format($combCosto,2).'</b></td></tr>
              </table>
              <div class="muted" style="margin-top:6px"><i>El fondo de combustible se asigna al chofer responsable.</i></div>
            </div>
          </div>

          <div class="box" style="margin-top:8px;">
            <b>Viáticos ('.htmlspecialchars((string)$viaticos['formula']).')</b>
            <table class="tbl">
              <thead><tr><th>Participante</th><th>Días</th><th style="text-align:right">Monto (USD)</th></tr></thead>
              <tbody>';
              foreach ($viaticos['detalles'] as $row) {
                $html .= '<tr><td>'.htmlspecialchars($row['nombre']).'</td><td style="text-align:center">'.(int)$row['dias'].'</td><td style="text-align:right">'.number_format((float)$row['monto'],2).'</td></tr>';
              }
        $html .= '</tbody>
              <tfoot><tr><th colspan="2" style="text-align:right">Total viáticos</th><th style="text-align:right">'.number_format((float)$viaticos['total_global'],2).'</th></tr></tfoot>
            </table>

            <div class="kpis">
              <div class="kpi"><b>Total Viáticos:</b><br>$'.number_format((float)$viaticos['total_global'],2).'</div>
              <div class="kpi"><b>Total Combustible:</b><br>$'.number_format((float)$combCosto,2).'</div>
              <div class="kpi total"><b>TOTAL GENERAL:</b><br><b>$'.number_format((float)$granTotal,2).'</b></div>
            </div>
          </div>

          <div class="box" style="margin-top:8px;">
            <b>Comentario del Presidente COMGA</b>
            <div class="muted" style="margin-top:6px">'.nl2br(htmlspecialchars($comentarioAlcalde)).'</div>
          </div>

          <div class="grid" style="margin-top:10px;">
            <div class="qr">
              '.$qrHtml.'
              <div class="muted" style="margin-top:6px">Escanee para validar · '.$verifyUrl.'</div>
            </div>
            <div class="box">
              <b>Declaración de validez</b>
              <div class="muted" style="margin-top:6px">Esta constancia se considera legítima si su código y URL de verificación coinciden con los emitidos por el Sistema de Gestión de Comisiones del <b>COMAGA</b>. La verificación pública permite a terceros confirmar su existencia y estado.</div>
              <div class="sig">
                <div class="line" style="margin:20px auto 6px;max-width:360px"></div>
                <div style="text-align:center;font-weight:600">Presidente COMGA</div>
              </div>
            </div>
          </div>

        </div></body></html>';

        // Render PDF y fallback HTML
        $pdfFs  = $dirFs . '/constancia.pdf';
        $pdfUrl = $this->buildPublicUrl('/constancias/comisiones/' . $comId . '/constancia.pdf');
        $htmlFs = $dirFs . '/constancia.html';

        try {
            $vendor = __DIR__ . '/../../vendor/autoload.php';
            if (file_exists($vendor)) {
                require_once $vendor;
                $dompdf = new \Dompdf\Dompdf(['isRemoteEnabled'=>true]);
                $dompdf->loadHtml($html);
                $dompdf->setPaper('A4', 'portrait');
                $dompdf->render();
                file_put_contents($pdfFs, $dompdf->output());
            } else {
                file_put_contents($htmlFs, $html);
                $pdfUrl = $this->buildPublicUrl('/constancias/comisiones/' . $comId . '/constancia.html');
            }
        } catch (\Throwable $e) {
            file_put_contents($htmlFs, $html);
            $pdfUrl = $this->buildPublicUrl('/constancias/comisiones/' . $comId . '/constancia.html');
        }

        // Actualiza paths + texto QR (URL de verificación)
        if (isset($const['id'])) {
            if (method_exists($mConst, 'updatePaths')) {
                $mConst->updatePaths((int)$const['id'], ['pdf_path'=>$pdfUrl, 'qr_text'=>$qrText]);
            } else {
                $db = DB::conn();
                $st = $db->prepare("UPDATE comision_constancias SET pdf_path=?, qr_text=? WHERE id=?");
                $st->execute([$pdfUrl, $qrText, (int)$const['id']]);
            }
        }

        (new Trazabilidad())->log('COMISIONES','CONSTANCIA_GEN', $alcalde['id'], $comId, $codigo);
    }
}
