<?php
declare(strict_types=1);

class DashboardController extends Controller
{
    /* ---------------------- Helpers de negocio ---------------------- */

    private function buildParticipantsList(string $participantesText, string $choferNombre = ''): array {
        $list = [];
        foreach (array_filter(array_map('trim', explode(',', (string)$participantesText))) as $c) {
            if ($c !== '') $list[] = $c;
        }
        $ch = trim($choferNombre);
        if ($ch !== '') $list[] = $ch;
        // únicos (case-insensitive)
        $seen = []; $out = [];
        foreach ($list as $n) {
            $k = mb_strtolower($n);
            if (!isset($seen[$k])) { $seen[$k]=1; $out[]=$n; }
        }
        return $out;
    }

    private function calcularViaticos(string $fini, string $ffin, array $nombres): array {
        try { $d1=new DateTime($fini); $d2=new DateTime($ffin); $dias=$d1->diff($d2)->days + 1; } catch (\Throwable $e) { $dias=1; }
        $total = 0.0;
        foreach ($nombres as $_) { $total += ($dias<=1) ? 40.0 : (80.0 * $dias); }
        return ['dias'=>$dias, 'total'=>$total];
    }

    private function calcularCombustible(?string $vehiculoNombre, ?float $kmTotales): array {
        $vehiculo = strtoupper(trim((string)$vehiculoNombre));
        $tabla = [
            'TOYOTA FORTUNES'      => ['tipo'=>'GASOLINA','km_gal'=>22.0],
            'TOYOTA HILUX'         => ['tipo'=>'GASOLINA','km_gal'=>36.0],
            'CHEVROLET LUV DIMAX'  => ['tipo'=>'DIÉSEL',  'km_gal'=>38.0],
            'VITARA 3 PUERTAS'     => ['tipo'=>'GASOLINA','km_gal'=>50.0],
        ];
        $kmPorGal = $tabla[$vehiculo]['km_gal'] ?? 30.0;
        $tipo     = $tabla[$vehiculo]['tipo']   ?? 'GASOLINA';
        $km = round(max(0.0, (float)$kmTotales), 2);

        $precioGas    = defined('FUEL_GAS_PRICE_USD')    ? (float)FUEL_GAS_PRICE_USD    : 2.88;
        $precioDiesel = defined('FUEL_DIESEL_PRICE_USD') ? (float)FUEL_DIESEL_PRICE_USD : 2.80;
        $precioGal = ($tipo === 'DIÉSEL') ? $precioDiesel : $precioGas;

        $gal = $km > 0 ? ($km / $kmPorGal) : 0.0;
        $costo = $gal * $precioGal;

        return ['tipo'=>$tipo, 'galones'=>$gal, 'precio_galon'=>$precioGal, 'costo_usd'=>$costo];
    }

    /** Devuelve serie de 12 puntos YYYY-MM => valor (rellenando ceros). */
    private function serie12(array $mapYmToValue): array {
        $serie = [];
        $cur = new DateTime('first day of this month'); $cur->modify('-11 months');
        for ($i=0; $i<12; $i++) {
            $k = $cur->format('Y-m');
            $serie[] = (float)($mapYmToValue[$k] ?? 0.0);
            $cur->modify('+1 month');
        }
        return $serie;
    }

    /* ------------------------------ Acción ------------------------------ */

    public function index(): void
    {
        require_login();
        $u = user();

        // Conexión
        if (!class_exists('DB')) { require_once __DIR__ . '/../config/Database.php'; }
        $db = DB::conn(); // asegura utf8mb4 en tu Database.php

        /* ======================== KPI PRINCIPALES ======================== */

        // Total comisiones
        $totalComisiones = (int)$db->query("SELECT COUNT(*) FROM comisiones WHERE deleted_at IS NULL")->fetchColumn();

        // ¿Existen columnas de estimados?
        $cols = $db->query("SHOW COLUMNS FROM comisiones")->fetchAll(PDO::FETCH_COLUMN);
        $tieneEstimados = in_array('viaticos_total_est', $cols, true)
                          && in_array('combustible_usd_est', $cols, true)
                          && in_array('total_general_est', $cols, true);

        $totalVia = 0.0; $totalComb = 0.0; $totalGen = 0.0;

        if ($tieneEstimados) {
            $sum = $db->query("
                SELECT
                  COALESCE(SUM(viaticos_total_est),0) AS via,
                  COALESCE(SUM(combustible_usd_est),0) AS comb,
                  COALESCE(SUM(total_general_est),0) AS tot
                FROM comisiones
                WHERE deleted_at IS NULL
            ")->fetch(PDO::FETCH_ASSOC);
            $totalVia  = (float)$sum['via'];
            $totalComb = (float)$sum['comb'];
            $totalGen  = (float)$sum['tot'];
        } else {
            // Fallback: calcula on-the-fly para cada comisión
            $rs = $db->query("
              SELECT fecha_inicio, fecha_fin, participantes_text, chofer_nombre,
                     vehiculo_nombre, total_km
              FROM comisiones
              WHERE deleted_at IS NULL
            ");
            foreach ($rs as $c) {
                $nombres = $this->buildParticipantsList((string)$c['participantes_text'], (string)$c['chofer_nombre']);
                $via = $this->calcularViaticos((string)$c['fecha_inicio'], (string)$c['fecha_fin'], $nombres);
                $comb = $this->calcularCombustible((string)$c['vehiculo_nombre'], (float)$c['total_km']);
                $totalVia  += (float)$via['total'];
                $totalComb += (float)$comb['costo_usd'];
                $totalGen  += (float)$via['total'] + (float)$comb['costo_usd'];
            }
        }

        // Comisiones del mes (por fecha_inicio)
        $comMes = (int)$db->query("
            SELECT COUNT(*) FROM comisiones
            WHERE deleted_at IS NULL
              AND MONTH(fecha_inicio)=MONTH(CURDATE())
              AND YEAR(fecha_inicio)=YEAR(CURDATE())
        ")->fetchColumn();

        // Tiempo promedio de aprobación (días)
        $horasProm = (float)$db->query("
            SELECT AVG(TIMESTAMPDIFF(HOUR, created_at, fecha_aprob))
            FROM comisiones
            WHERE deleted_at IS NULL AND fecha_aprob IS NOT NULL
        ")->fetchColumn();
        $diasProm = $horasProm > 0 ? round($horasProm/24, 1) : 0.0;

        $kpis = [
            ['label'=>'Total comisiones','value'=>$totalComisiones,'unit'=>'','hint'=>'Histórico'],
            ['label'=>'Total viáticos','value'=>number_format($totalVia,2),'unit'=>'USD','hint'=>'Acumulado'],
            ['label'=>'Fondo combustible','value'=>number_format($totalComb,2),'unit'=>'USD','hint'=>'Acumulado'],
            ['label'=>'Total general','value'=>number_format($totalGen,2),'unit'=>'USD','hint'=>'Viáticos + combustible'],
            ['label'=>'Comisiones del mes','value'=>$comMes,'unit'=>'','hint'=>'Mes en curso'],
            ['label'=>'Tiempo prom. aprobación','value'=>$diasProm,'unit'=>'d','hint'=>'Promedio'],
        ];

        /* ======================== SERIES (12 meses) ======================== */

        // Comisiones por mes (conteo)
        $rows = $db->query("
            SELECT DATE_FORMAT(fecha_inicio,'%Y-%m') ym, COUNT(*) v
            FROM comisiones
            WHERE deleted_at IS NULL
              AND fecha_inicio >= DATE_SUB(CURDATE(), INTERVAL 12 MONTH)
            GROUP BY ym
            ORDER BY ym
        ")->fetchAll(PDO::FETCH_KEY_PAIR) ?: [];
        $tr_comisiones = $this->serie12($rows);

        // Gasto (viáticos + combustible) por mes
        if ($tieneEstimados) {
            $rows = $db->query("
              SELECT DATE_FORMAT(fecha_inicio,'%Y-%m') ym,
                     SUM(COALESCE(viaticos_total_est,0) + COALESCE(combustible_usd_est,0)) v
              FROM comisiones
              WHERE deleted_at IS NULL
                AND fecha_inicio >= DATE_SUB(CURDATE(), INTERVAL 12 MONTH)
              GROUP BY ym
              ORDER BY ym
            ")->fetchAll(PDO::FETCH_KEY_PAIR) ?: [];
            $tr_gasto = $this->serie12($rows);
        } else {
            // Fallback: recalcular por comisión del último año y agrupar en PHP
            $rows = $db->query("
              SELECT fecha_inicio, fecha_fin, participantes_text, chofer_nombre, vehiculo_nombre, total_km
              FROM comisiones
              WHERE deleted_at IS NULL
                AND fecha_inicio >= DATE_SUB(CURDATE(), INTERVAL 12 MONTH)
            ")->fetchAll(PDO::FETCH_ASSOC) ?: [];
            $map = [];
            foreach ($rows as $c) {
                $ym = date('Y-m', strtotime($c['fecha_inicio']));
                $nombres = $this->buildParticipantsList((string)$c['participantes_text'], (string)$c['chofer_nombre']);
                $via = $this->calcularViaticos((string)$c['fecha_inicio'], (string)$c['fecha_fin'], $nombres);
                $comb = $this->calcularCombustible((string)$c['vehiculo_nombre'], (float)$c['total_km']);
                $map[$ym] = ($map[$ym] ?? 0) + (float)$via['total'] + (float)$comb['costo_usd'];
            }
            $tr_gasto = $this->serie12($map);
        }

        $trends = [
            'comisiones_mes' => $tr_comisiones,
            'gasto_mes'      => $tr_gasto,
        ];

        /* =================== DESGLOSE / TOP ÁREAS / RECIENTE =================== */

        $estado = $db->query("
            SELECT estado AS nombre, COUNT(*) AS valor
            FROM comisiones
            WHERE deleted_at IS NULL
            GROUP BY estado
            ORDER BY valor DESC
        ")->fetchAll(PDO::FETCH_ASSOC) ?: [];

        $topAreas = $db->query("
            SELECT a.nombre AS area,
                   COUNT(*) AS total,
                   CONCAT(ROUND(AVG(TIMESTAMPDIFF(HOUR, c.created_at, c.fecha_aprob))/24,1),' d') AS tiempo_prom
            FROM comisiones c
            LEFT JOIN areas a ON a.id = c.area_id
            WHERE c.deleted_at IS NULL
              AND c.created_at >= (CURRENT_DATE - INTERVAL 30 DAY)
            GROUP BY a.id, a.nombre
            ORDER BY total DESC
            LIMIT 5
        ")->fetchAll(PDO::FETCH_ASSOC) ?: [];

        // Actividad reciente (tabla trazabilidad tal como está en tu dump)
        $recent = $db->query("
            SELECT t.modulo, t.accion, u.nombre AS usuario, t.detalle, t.fecha_hora
            FROM trazabilidad t
            LEFT JOIN users u ON u.id = t.usuario_id
            ORDER BY t.id DESC
            LIMIT 12
        ")->fetchAll(PDO::FETCH_ASSOC) ?: [];

        $recent = array_map(function($r){
            return [
                'accion'  => (string)$r['accion'],
                'modulo'  => (string)$r['modulo'],
                'usuario' => (string)($r['usuario'] ?? '—'),
                'estado'  => (string)($r['detalle'] ?? '—'),
                'fecha'   => date('d/m H:i', strtotime((string)$r['fecha_hora'])),
            ];
        }, $recent);

        // Render
        $this->view('dashboard/index', compact('u','kpis','trends','estado','recent','topAreas'));
    }
}
