<?php
declare(strict_types=1);

require_once __DIR__ . '/../models/Reportes.php';

class ReportesController extends Controller {

    /** Solo ADMIN y ALCALDE pueden ver reportes */
    public function index(): void { require_login();
        $u = user();
        $rol = strtoupper($u['rol'] ?? '');
        if (!in_array($rol, ['ADMIN','ALCALDE'], true)) {
            http_response_code(403); exit('No autorizado');
        }

        // ===== Filtros =====
        $q       = trim($_GET['q']   ?? '');         // busca en título y descripción
        $chofer  = trim($_GET['ch']  ?? '');         // chofer exacto o parcial
        $desde   = trim($_GET['d']   ?? '');         // YYYY-MM-DD
        $hasta   = trim($_GET['h']   ?? '');         // YYYY-MM-DD
        $areaId  = (int)($_GET['a']  ?? 0);          // opcional
        $estado  = trim($_GET['e']   ?? '');         // APROBADA, RECHAZADA, etc.

        $model = new Reportes();
        $rows  = $model->comisionesConDatos($q, $chofer, $desde, $hasta, $areaId, $estado);

        // Cálculos de totales
        $totViaticos   = 0.0;
        $totCombustible= 0.0;
        $totGeneral    = 0.0;
        foreach ($rows as &$r) {
            $totViaticos    += (float)$r['viaticos_total'];
            $totCombustible += (float)$r['combustible_costo'];
            $r['gran_total'] = (float)$r['viaticos_total'] + (float)$r['combustible_costo'];
            $totGeneral     += (float)$r['gran_total'];
        } unset($r);

        // Top por chofer / por comisión en el rango
        $tops = $model->tops($q, $chofer, $desde, $hasta, $areaId, $estado);

        // Export CSV
        if (isset($_GET['export']) && $_GET['export'] === 'csv') {
            $this->exportCsv($rows, 'reportes_comisiones_' . date('Ymd_His') . '.csv');
            return;
        }

        $this->view('reportes/index', [
            'rows' => $rows,
            'totViaticos' => $totViaticos,
            'totCombustible' => $totCombustible,
            'totGeneral' => $totGeneral,
            'tops' => $tops,
            // filtros
            'q'=>$q,'chofer'=>$chofer,'desde'=>$desde,'hasta'=>$hasta,'areaId'=>$areaId,'estado'=>$estado
        ]);
    }


// ==== Listado / búsqueda por funcionario (ADMIN y ALCALDE) ====
    public function funcionario(): void { require_login();
        $u = user(); $rol = strtoupper($u['rol'] ?? '');
        if (!in_array($rol, ['ADMIN','ALCALDE'], true)) { http_response_code(403); exit('No autorizado'); }

        $qNombre   = trim($_GET['q'] ?? '');
        $desde     = trim($_GET['desde'] ?? '');
        $hasta     = trim($_GET['hasta'] ?? '');
        $chofer    = trim($_GET['chofer'] ?? '');

        // Trae comisiones base (ajusta nombres de tabla/campos según tu DB)
        $db = DB::conn();
        $sql = "SELECT c.id, c.titulo, c.descripcion, c.fecha_inicio, c.fecha_fin,
                       c.participantes_text, c.chofer_nombre, c.vehiculo_nombre,
                       c.origen_ciudad, c.destino_ida_ciudad, c.destino_retorno_ciudad,
                       c.total_km
                FROM comisiones c
                WHERE 1=1";
        $params = [];

        if ($desde !== '') { $sql .= " AND c.fecha_inicio >= ?"; $params[] = $desde; }
        if ($hasta !== '') { $sql .= " AND c.fecha_fin     <= ?"; $params[] = $hasta; }

        if ($qNombre !== '') {
            $like = '%'.$qNombre.'%';
            $sql .= " AND (c.participantes_text LIKE ? OR c.titulo LIKE ? OR c.descripcion LIKE ?)";
            $params[]=$like; $params[]=$like; $params[]=$like;
        }

        if ($chofer !== '') {
            $sql .= " AND c.chofer_nombre = ?";
            $params[] = $chofer;
        }

        $sql .= " ORDER BY c.fecha_inicio DESC LIMIT 1000";
        $stmt = $db->prepare($sql);
        $stmt->execute($params);
        $rows = $stmt->fetchAll();

        // Construir resumen por funcionario (viáticos + fondo combustible solo cuando es chofer)
        $porFuncionario = [];   // key => ['nombre','viaticos','fondo','total','items'=>[]]

        foreach ($rows as $r) {
            // Lista de nombres declarados en la comisión
            $nombres = [];
            $chips = array_filter(array_map('trim', explode(',', (string)($r['participantes_text'] ?? ''))));
            foreach ($chips as $c) if ($c!=='') $nombres[] = $c;

            // Combustible de la comisión (fondo si la persona fue chofer)
            $comb = $this->calcularCombustible((string)$r['vehiculo_nombre'], (float)$r['total_km']);
            $fondoCombChofer = (float)$comb['costo_usd'];

            // Viáticos por persona + fondo al chofer
            $det = $this->viaticosPorPersonaConFondo(
                (string)$r['fecha_inicio'], (string)$r['fecha_fin'], $nombres, (string)$r['chofer_nombre'], $fondoCombChofer
            );

            foreach ($det['detalles'] as $p) {
                $key = mb_strtolower(trim($p['nombre']));
                if (!isset($porFuncionario[$key])) {
                    $porFuncionario[$key] = [
                        'nombre' => $p['nombre'],
                        'viaticos' => 0.0,
                        'fondo'   => 0.0,
                        'total'   => 0.0,
                        'items'   => []
                    ];
                }
                $porFuncionario[$key]['viaticos'] += $p['viaticos_usd'];
                $porFuncionario[$key]['fondo']    += $p['fondo_comb_usd'];
                $porFuncionario[$key]['total']    += $p['total_persona'];

                $porFuncionario[$key]['items'][] = [
                    'comision_id' => (int)$r['id'],
                    'titulo'      => (string)$r['titulo'],
                    'descripcion' => (string)$r['descripcion'],
                    'rango'       => (string)$r['fecha_inicio'].' → '.(string)$r['fecha_fin'],
                    'origen'      => (string)$r['origen_ciudad'],
                    'destino'     => (string)$r['destino_ida_ciudad'],
                    'es_chofer'   => (mb_strtolower(trim($p['nombre'])) === mb_strtolower(trim((string)$r['chofer_nombre']))),
                    'viaticos'    => (float)$p['viaticos_usd'],
                    'fondo'       => (float)$p['fondo_comb_usd'],
                    'total'       => (float)$p['total_persona'],
                ];
            }
        }

        // Pasar de mapa a lista y ordenar por total desc
        $data = array_values($porFuncionario);
        usort($data, fn($a,$b) => $b['total'] <=> $a['total']);

        $this->view('reportes/funcionario', [
            'data'   => $data,
            'q'      => $qNombre,
            'desde'  => $desde,
            'hasta'  => $hasta,
            'chofer' => $chofer
        ]);
    }

    // ==== PDF individual del funcionario seleccionado ====
    // Ruta sugerida: GET /reportes/funcionario_pdf?nombre=...&desde=...&hasta=...
    public function funcionario_pdf(): void { require_login();
        $u = user(); $rol = strtoupper($u['rol'] ?? '');
        if (!in_array($rol, ['ADMIN','ALCALDE'], true)) { http_response_code(403); exit('No autorizado'); }

        $nombre   = trim($_GET['nombre'] ?? '');
        if ($nombre === '') { http_response_code(400); exit('Falta parámetro nombre'); }
        $desde    = trim($_GET['desde'] ?? '');
        $hasta    = trim($_GET['hasta'] ?? '');

        // 1) Reutilizamos la misma consulta de funcionario(), pero luego filtramos los items del NOMBRE solicitado
        $db = DB::conn();
        $sql = "SELECT c.id, c.titulo, c.descripcion, c.fecha_inicio, c.fecha_fin,
                       c.participantes_text, c.chofer_nombre, c.vehiculo_nombre,
                       c.origen_ciudad, c.destino_ida_ciudad, c.destino_retorno_ciudad,
                       c.total_km
                FROM comisiones c
                WHERE 1=1";
        $params = [];
        if ($desde !== '') { $sql .= " AND c.fecha_inicio >= ?"; $params[] = $desde; }
        if ($hasta !== '') { $sql .= " AND c.fecha_fin     <= ?"; $params[] = $hasta; }
        // por performance, si quieres también filtrar por LIKE en participantes
        $like = '%'.$nombre.'%';
        $sql .= " AND (c.participantes_text LIKE ? OR c.chofer_nombre = ?)";
        $params[] = $like;
        $params[] = $nombre;

        $sql .= " ORDER BY c.fecha_inicio DESC LIMIT 2000";
        $st = $db->prepare($sql); $st->execute($params);
        $rows = $st->fetchAll();

        $items = [];
        $sumVia = 0.0; $sumFondo = 0.0; $sumTot = 0.0;

        foreach ($rows as $r) {
            // nombres en la comisión
            $nombres = [];
            $chips = array_filter(array_map('trim', explode(',', (string)($r['participantes_text'] ?? ''))));
            foreach ($chips as $c) if ($c!=='') $nombres[] = $c;

            // Combustible de la comisión
            $comb = $this->calcularCombustible((string)$r['vehiculo_nombre'], (float)$r['total_km']);
            $fondoCombChofer = (float)$comb['costo_usd'];

            // Detalle por persona
            $det = $this->viaticosPorPersonaConFondo(
                (string)$r['fecha_inicio'], (string)$r['fecha_fin'], $nombres, (string)$r['chofer_nombre'], $fondoCombChofer
            );

            // Buscar solo el registro del funcionario $nombre
            foreach ($det['detalles'] as $p) {
                if (mb_strtolower(trim($p['nombre'])) !== mb_strtolower($nombre)) continue;

                $esChofer = (mb_strtolower(trim($p['nombre'])) === mb_strtolower(trim((string)$r['chofer_nombre'])));

                $row = [
                    'comision_id' => (int)$r['id'],
                    'titulo'      => (string)$r['titulo'],
                    'descripcion' => (string)$r['descripcion'],
                    'rango'       => (string)$r['fecha_inicio'].' → '.(string)$r['fecha_fin'],
                    'origen'      => (string)$r['origen_ciudad'],
                    'destino'     => (string)$r['destino_ida_ciudad'],
                    'es_chofer'   => $esChofer,
                    'viaticos'    => (float)$p['viaticos_usd'],
                    'fondo'       => (float)$p['fondo_comb_usd'],
                    'total'       => (float)$p['total_persona'],
                ];
                $items[] = $row;
                $sumVia  += $row['viaticos'];
                $sumFondo+= $row['fondo'];
                $sumTot  += $row['total'];
            }
        }

        // 2) Construir HTML del PDF
        $logo = (defined('BASE_URL') ? rtrim(BASE_URL,'/') : '') . '/img/logoc.png';
        $periodo = ($desde || $hasta) ? (h($desde ?: '…').' a '.h($hasta ?: '…')) : 'Todo';

        $tblRows = '';
        foreach ($items as $it) {
            $tblRows .= '<tr>
              <td>'.htmlspecialchars($it['titulo']).'</td>
              <td>'.htmlspecialchars($it['rango']).'</td>
              <td>'.htmlspecialchars($it['origen']).'</td>
              <td>'.htmlspecialchars($it['destino']).'</td>
              <td>'.($it['es_chofer']?'<b>Chofer</b>':'Participante').'</td>
              <td style="text-align:right">'.number_format($it['viaticos'],2).'</td>
              <td style="text-align:right">'.number_format($it['fondo'],2).'</td>
              <td style="text-align:right"><b>'.number_format($it['total'],2).'</b></td>
            </tr>';
        }
        if ($tblRows === '') {
            $tblRows = '<tr><td colspan="8" style="text-align:center;color:#6b7280">Sin registros para el periodo</td></tr>';
        }

        $html = '
        <html><head><meta charset="utf-8"><style>
          @page { margin: 24px; }
          body{ font-family: DejaVu Sans, sans-serif; font-size:12px; color:#111827; }
          .hdr{ display:flex; align-items:center; gap:12px; margin-bottom:8px; }
          .logo{ width:42px; height:42px; border:1px solid #cbd5e1; border-radius:8px; overflow:hidden; }
          .title{ font-size:18px; font-weight:700; color:#1e3a8a; }
          .muted{ color:#6b7280; }
          .kpis{ display:grid; grid-template-columns:1fr 1fr 1fr; gap:10px; margin:10px 0; }
          .kpi{ border:1px solid #e5e7eb; border-radius:10px; padding:8px; }
          .tbl{ width:100%; border-collapse:collapse; }
          .tbl th,.tbl td{ border:1px solid #e5e7eb; padding:6px; }
          .tbl th{ background:#f8fafc; text-align:left; }
        </style></head><body>
          <div class="hdr">
            <div class="logo"><img src="'.$logo.'" alt="Logo" style="width:100%;height:100%;object-fit:cover"/></div>
            <div>
              <div class="title">Reporte de viáticos por funcionario</div>
              <div class="muted">Funcionario: <b>'.htmlspecialchars($nombre).'</b> · Periodo: '.$periodo.'</div>
            </div>
          </div>

          <div class="kpis">
            <div class="kpi"><div class="muted">Total viáticos</div><div class="h4">$'.number_format($sumVia,2).'</div></div>
            <div class="kpi"><div class="muted">Fondo combustible (como chofer)</div><div class="h4">$'.number_format($sumFondo,2).'</div></div>
            <div class="kpi"><div class="muted">Total general</div><div class="h4"><b>$'.number_format($sumTot,2).'</b></div></div>
          </div>

          <table class="tbl">
            <thead>
              <tr>
                <th>Comisión</th>
                <th>Rango</th>
                <th>Origen</th>
                <th>Destino</th>
                <th>Rol</th>
                <th style="text-align:right">Viáticos</th>
                <th style="text-align:right">Fondo comb.</th>
                <th style="text-align:right">Total</th>
              </tr>
            </thead>
            <tbody>'.$tblRows.'</tbody>
          </table>

          <div class="muted" style="margin-top:10px">
            * El fondo de combustible se asigna exclusivamente cuando el funcionario actúa como chofer de la comisión.
          </div>
        </body></html>';

        // 3) Renderizar y enviar PDF
        try {
            $vendor = __DIR__ . '/../../vendor/autoload.php';
            if (file_exists($vendor)) { require_once $vendor; }
            $dompdf = new \Dompdf\Dompdf(['isRemoteEnabled'=>true]);
            $dompdf->loadHtml($html);
            $dompdf->setPaper('A4', 'portrait');
            $dompdf->render();

            $filename = 'reporte_funcionario_'.preg_replace('/[^A-Za-z0-9_-]+/','_', $nombre).'_'.date('Ymd_His').'.pdf';
            header('Content-Type: application/pdf');
            header('Content-Disposition: inline; filename="'.$filename.'"');
            echo $dompdf->output(); exit;
        } catch (\Throwable $e) {
            // Fallback a HTML si Dompdf no está disponible
            header('Content-Type: text/html; charset=utf-8');
            echo $html; exit;
        }
    }

    /* ===== Helpers internos (mismos que ya venías usando) ===== */

    private function calcularCombustible(?string $vehiculoNombre, ?float $kmTotales): array {
        $vehiculo = strtoupper(trim((string)$vehiculoNombre));
        $tabla = [
            'TOYOTA FORTUNES'      => ['tipo' => 'GASOLINA', 'km_gal' => 22.0],
            'TOYOTA HILUX'         => ['tipo' => 'GASOLINA', 'km_gal' => 36.0],
            'CHEVROLET LUV DIMAX'  => ['tipo' => 'DIÉSEL',   'km_gal' => 38.0],
            'VITARA 3 PUERTAS'     => ['tipo' => 'GASOLINA', 'km_gal' => 50.0],
        ];
        $kmPorGal = $tabla[$vehiculo]['km_gal'] ?? 30.0;
        $tipo     = $tabla[$vehiculo]['tipo']   ?? 'GASOLINA';
        $km = round(max(0.0, (float)$kmTotales), 2);

        $precioGas    = defined('FUEL_GAS_PRICE_USD')    ? (float)FUEL_GAS_PRICE_USD    : 2.88;
        $precioDiesel = defined('FUEL_DIESEL_PRICE_USD') ? (float)FUEL_DIESEL_PRICE_USD : 2.80;
        $precioGal = ($tipo === 'DIÉSEL') ? $precioDiesel : $precioGas;

        $galones = $km > 0 ? ($km / $kmPorGal) : 0.0;
        $costo = $galones * $precioGal;

        return [
            'tipo'=>$tipo,'km_por_galon'=>$kmPorGal,'galones'=>$galones,
            'precio_galon'=>$precioGal,'costo_usd'=>$costo,
            'km_por_litro'=>($kmPorGal/3.785411784),'litros'=>($galones*3.785411784),
            'precio_litro'=>($precioGal/3.785411784),
        ];
    }

    /** Viáticos por persona + fondo combustible SOLO para el chofer */
    private function viaticosPorPersonaConFondo(
        string $fini, string $ffin, array $nombres, string $choferNombre, float $combustibleUSD
    ): array {
        try { $d1 = new DateTime($fini); $d2 = new DateTime($ffin); $dias = $d1->diff($d2)->days + 1; }
        catch (\Throwable $e) { $dias = 1; }
        $montoViatico = ($dias <= 1) ? 40.0 : (80.0 * $dias);
        $choferKey = mb_strtolower(trim($choferNombre));

        $out=[]; $total=0.0; $seen=[];
        foreach ($nombres as $nom) {
            $k=mb_strtolower(trim($nom));
            if ($k==='') continue;
            if (isset($seen[$k])) continue; $seen[$k]=1;

            $via  = $montoViatico;
            $fond = ($k === $choferKey) ? max(0.0, (float)$combustibleUSD) : 0.0;
            $sum  = $via + $fond;

            $out[] = [
                'nombre'         => $nom,
                'dias'           => $dias,
                'viaticos_usd'   => $via,
                'fondo_comb_usd' => $fond,
                'total_persona'  => $sum,
            ];
            $total += $sum;
        }
        return ['dias'=>$dias,'detalles'=>$out,'total_global'=>$total,'formula'=>($dias<=1)?'1 día × USD 40':($dias.' días × USD 80')];
    }
    /* ================= helpers ================= */

    private function exportCsv(array $rows, string $filename): void {
        header('Content-Type: text/csv; charset=UTF-8');
        header('Content-Disposition: attachment; filename="'.$filename.'"');
        $out = fopen('php://output','w');
        // BOM UTF-8 para Excel
        fwrite($out, "\xEF\xBB\xBF");
        fputcsv($out, [
            'ID','Título','Rango', 'Área', 'Chofer', 'Vehículo', 'Km totales',
            'Participantes (#)', 'Días', 'Viáticos USD', 'Combustible USD', 'Total USD', 'Estado'
        ]);

        foreach ($rows as $r) {
            fputcsv($out, [
                $r['id'],
                $r['titulo'],
                $r['fecha_inicio'].' → '.$r['fecha_fin'],
                $r['area_nombre'],
                $r['chofer_nombre'],
                $r['vehiculo_nombre'],
                number_format((float)$r['total_km'],2,'.',''),
                (int)$r['participantes_num'],
                (int)$r['dias'],
                number_format((float)$r['viaticos_total'],2,'.',''),
                number_format((float)$r['combustible_costo'],2,'.',''),
                number_format((float)$r['gran_total'],2,'.',''),
                strtoupper($r['estado'] ?? '')
            ]);
        }
        fclose($out);
        exit;
    }
}
