<?php
declare(strict_types=1);

class TrazabilidadController extends Controller {

    /** Solo ADMIN y ALCALDE */
    private function requireAdminOrAlcalde(): array {
        require_login();
        $u = user();
        $rol = strtoupper($u['rol'] ?? '');
        if (!in_array($rol, ['ADMIN','ALCALDE'], true)) {
            http_response_code(403);
            exit('No autorizado');
        }
        return $u;
    }

    public function index(): void {
        $u = $this->requireAdminOrAlcalde();

        // ===== Filtros (GET) =====
        $modulo = trim($_GET['mod'] ?? 'COMISIONES');
        $q      = trim($_GET['q'] ?? '');
        $accion = trim($_GET['accion'] ?? '');
        $usr    = trim($_GET['usuario'] ?? '');
        $desde  = trim($_GET['desde'] ?? '');
        $hasta  = trim($_GET['hasta'] ?? '');
        $limit  = max(50, min(2000, (int)($_GET['lim'] ?? 800))); // seguridad

        // Obtener desde el modelo (no lo tocamos)
        $m = new Trazabilidad();
        // byModulo(modulo, limit) ya lo usabas; si no existe limit alto, igual nos quedamos con lo que devuelva
        $rows = $m->byModulo($modulo, $limit);

        // Normaliza fechas a timestamps (00:00:00 y 23:59:59)
        $tsDesde = null; $tsHasta = null;
        if ($desde !== '') { try { $tsDesde = (new DateTime($desde.' 00:00:00'))->getTimestamp(); } catch(\Throwable $e){} }
        if ($hasta !== '') { try { $tsHasta = (new DateTime($hasta.' 23:59:59'))->getTimestamp(); } catch(\Throwable $e){} }

        // ===== Filtrado en memoria =====
        $logs = array_values(array_filter($rows, function($r) use ($q,$accion,$usr,$tsDesde,$tsHasta){
            $ok = true;

            if ($q !== '') {
                $blob = mb_strtolower(
                    (string)($r['modulo']??'').' '.
                    (string)($r['accion']??'').' '.
                    (string)($r['usuario']??'').' '.
                    (string)($r['detalle']??'').' '.
                    (string)($r['ref_id']??'')
                );
                $ok = $ok && (strpos($blob, mb_strtolower($q)) !== false);
            }
            if ($accion !== '') {
                $ok = $ok && (mb_strtoupper((string)($r['accion']??'')) === mb_strtoupper($accion));
            }
            if ($usr !== '') {
                $ok = $ok && (mb_strtolower((string)($r['usuario']??'')) === mb_strtolower($usr));
            }

            if ($tsDesde !== null || $tsHasta !== null) {
                $fh = (string)($r['fecha_hora'] ?? '');
                $ts = @strtotime($fh);
                if ($ts !== false) {
                    if ($tsDesde !== null) $ok = $ok && ($ts >= $tsDesde);
                    if ($tsHasta !== null) $ok = $ok && ($ts <= $tsHasta);
                }
            }

            return $ok;
        }));

        // ===== Estadísticas para los KPIs / selects =====
        $total = count($logs);
        $h24 = time() - 86400;
        $ult24 = 0;
        $accCounts = [];
        $usrCounts = [];
        foreach ($logs as $r) {
            $acc = strtoupper((string)($r['accion'] ?? 'OTRO'));
            $usrn= (string)($r['usuario'] ?? '—');
            $accCounts[$acc] = ($accCounts[$acc] ?? 0) + 1;
            $usrCounts[$usrn] = ($usrCounts[$usrn] ?? 0) + 1;
            $ts = @strtotime((string)($r['fecha_hora'] ?? ''));
            if ($ts && $ts >= $h24) $ult24++;
        }
        // Top acciones (ordenadas por frecuencia)
        arsort($accCounts);
        // Lista ordenada de usuarios
        ksort($usrCounts, SORT_NATURAL | SORT_FLAG_CASE);

        // ===== Exportar CSV si se solicita =====
        if (isset($_GET['export']) && $_GET['export'] === 'csv') {
            header('Content-Type: text/csv; charset=UTF-8');
            header('Content-Disposition: attachment; filename="trazabilidad_'.$modulo.'_'.date('Ymd_His').'.csv"');
            $out = fopen('php://output', 'w');
            // BOM para Excel
            fprintf($out, chr(0xEF).chr(0xBB).chr(0xBF));
            fputcsv($out, ['ID','Fecha/Hora','Módulo','Acción','Usuario','Ref','Detalle']);
            foreach ($logs as $r) {
                fputcsv($out, [
                    (int)($r['id'] ?? 0),
                    (string)($r['fecha_hora'] ?? ''),
                    (string)($r['modulo'] ?? ''),
                    (string)($r['accion'] ?? ''),
                    (string)($r['usuario'] ?? ''),
                    (string)($r['ref_id'] ?? ''),
                    (string)($r['detalle'] ?? ''),
                ]);
            }
            fclose($out);
            exit;
        }

        // Render
        $this->view('trazabilidad/index', [
            'logs'      => $logs,
            'modulo'    => $modulo,
            'q'         => $q,
            'accion'    => $accion,
            'usuario'   => $usr,
            'desde'     => $desde,
            'hasta'     => $hasta,
            'accCounts' => $accCounts,
            'usrCounts' => $usrCounts,
            'stats'     => ['total'=>$total, 'ult24'=>$ult24],
        ]);
    }
}
