<?php
class Comision extends Model {

    /** Listado clásico (sin constancia), compatibilidad */
    public function all(): array {
        $sql = "SELECT c.*, a.nombre AS area_nombre, d.nombre AS depto_nombre, u.nombre AS creador_nombre
                  FROM comisiones c
             LEFT JOIN areas a ON a.id=c.area_id
             LEFT JOIN departamentos d ON d.id=c.department_id
             LEFT JOIN users u ON u.id=c.creador_id
                 WHERE c.deleted_at IS NULL
              ORDER BY c.id DESC";
        return $this->db->query($sql)->fetchAll();
    }

    /** Trae una comisión por ID (sin constancia) */
    public function find(int $id): ?array {
        $stmt = $this->db->prepare("SELECT * FROM comisiones WHERE id=? AND deleted_at IS NULL");
        $stmt->execute([$id]);
        $row = $stmt->fetch();
        return $row ?: null;
    }

    /**
     * Crea comisión (+ guarda estimados)
     * Campos nuevos: dias_viatico, viaticos_total_est, combustible_usd_est, chofer_total_est, total_general_est
     */
    public function create(array $data): int {
        $stmt = $this->db->prepare("
            INSERT INTO comisiones
                (titulo, descripcion, fecha_inicio, fecha_fin,
                 area_id, department_id,
                 participantes_text,
                 chofer_nombre, vehiculo_nombre,
                 origen_ciudad, destino_ida_ciudad, destino_retorno_ciudad, total_km,
                 dias_viatico, viaticos_total_est, combustible_usd_est, chofer_total_est, total_general_est,
                 estado, creador_id, created_at)
            VALUES
                (?,?,?,?,?,
                 ?,?,
                 ?,?,
                 ?,?,?,?,
                 ?,?,?,?,?,
                 'PENDIENTE', ?, NOW())
        ");

        $stmt->execute([
            $data['titulo'],
            $data['descripcion'],
            $data['fecha_inicio'],
            $data['fecha_fin'],

            $data['area_id'] ?: null,
            $data['department_id'] ?: null,

            $data['participantes_text'] ?: null,

            $data['chofer_nombre'] ?: null,
            $data['vehiculo_nombre'] ?: null,

            $data['origen_ciudad'] ?: null,
            $data['destino_ida_ciudad'] ?: null,
            $data['destino_retorno_ciudad'] ?: null,
            isset($data['total_km']) ? (float)$data['total_km'] : null,

            $data['dias_viatico'] ?? null,
            $data['viaticos_total_est'] ?? null,
            $data['combustible_usd_est'] ?? null,
            $data['chofer_total_est'] ?? null,
            $data['total_general_est'] ?? null,

            $data['creador_id']
        ]);

        $id = (int)$this->db->lastInsertId();

        // Participantes del sistema (opcional)
        if (!empty($data['participantes']) && is_array($data['participantes'])) {
            $ins = $this->db->prepare("INSERT INTO comision_participantes (comision_id, usuario_id) VALUES (?, ?)");
            foreach ($data['participantes'] as $uid) {
                $uid = (int)$uid;
                if ($uid > 0) $ins->execute([$id, $uid]);
            }
        }

        return $id;
    }

    /** Anexa/actualiza metadatos del archivo de planificación */
    public function anexarPlan(int $id, array $plan): bool {
        $stmt = $this->db->prepare("
            UPDATE comisiones
               SET plan_nombre=?, plan_ruta=?, plan_mime=?, plan_tamano=?
             WHERE id=?");
        return $stmt->execute([$plan['nombre'], $plan['ruta'], $plan['mime'], (int)$plan['tamano'], $id]);
    }

    /** Cambia estado (APROBADA/RECHAZADA) + comentario y aprobador */
    public function updateEstado(int $id, string $estado, string $comentario, int $aprobadorId): bool {
        $stmt = $this->db->prepare("
            UPDATE comisiones
               SET estado=?, comentario_aprob=?, aprobador_id=?, fecha_aprob=NOW()
             WHERE id=?");
        return $stmt->execute([$estado, $comentario, $aprobadorId, $id]);
    }

    /* ============ Listados con última constancia ============ */

    private function baseQueryWithConst(): string {
        return "
        SELECT
            c.*,
            a.nombre AS area_nombre,
            d.nombre AS depto_nombre,
            u.nombre AS creador_nombre,
            cc.codigo       AS const_codigo,
            cc.pdf_path     AS const_pdf,
            cc.firmado_path AS const_firmado
          FROM comisiones c
     LEFT JOIN areas a ON a.id = c.area_id
     LEFT JOIN departamentos d ON d.id = c.department_id
     LEFT JOIN users u ON u.id = c.creador_id
     LEFT JOIN (
            SELECT t.*
              FROM comision_constancias t
        INNER JOIN (SELECT MAX(id) AS id FROM comision_constancias GROUP BY comision_id) ult
                ON ult.id = t.id
        ) cc ON cc.comision_id = c.id
         WHERE c.deleted_at IS NULL
        ";
    }

    /** Todas (ADMIN/ALCALDE/PRESIDENTE) */
    public function allWithConst(): array {
        $sql = $this->baseQueryWithConst() . " ORDER BY c.id DESC";
        return $this->db->query($sql)->fetchAll();
    }

    /** Solo creador */
    public function allByCreadorWithConst(int $userId): array {
        $sql = $this->baseQueryWithConst() . " AND c.creador_id = ? ORDER BY c.id DESC";
        $st = $this->db->prepare($sql);
        $st->execute([$userId]);
        return $st->fetchAll();
    }
}
