<?php
declare(strict_types=1);

class Reportes {

    /** Lee comisiones + compone participantes y cálculos de negocio */
    public function comisionesConDatos(string $q, string $chofer, string $desde, string $hasta, int $areaId, string $estado): array {
        $db = DB::conn();

        // Base query
        $sql = "SELECT c.id, c.titulo, c.descripcion, c.fecha_inicio, c.fecha_fin, c.estado,
                       c.chofer_nombre, c.vehiculo_nombre, c.total_km,
                       a.nombre AS area_nombre
                FROM comisiones c
                LEFT JOIN areas a ON a.id = c.area_id
                WHERE 1=1";
        $params = [];

        if ($q !== '') {
            $sql .= " AND (c.titulo LIKE ? OR c.descripcion LIKE ?)";
            $params[] = "%$q%"; $params[] = "%$q%";
        }
        if ($chofer !== '') {
            $sql .= " AND c.chofer_nombre LIKE ?";
            $params[] = "%$chofer%";
        }
        if ($desde !== '') { $sql .= " AND c.fecha_inicio >= ?"; $params[] = $desde; }
        if ($hasta !== '') { $sql .= " AND c.fecha_fin <= ?";  $params[] = $hasta; }
        if ($areaId > 0)   { $sql .= " AND c.area_id = ?";     $params[] = $areaId; }
        if ($estado !== ''){ $sql .= " AND c.estado = ?";      $params[] = $estado; }

        $sql .= " ORDER BY c.fecha_inicio DESC, c.id DESC LIMIT 1000";

        $st = $db->prepare($sql);
        $st->execute($params);
        $rows = $st->fetchAll();

        if (!$rows) return [];

        // Cargar participantes del sistema por bloque
        $ids = array_column($rows, 'id');
        $parts = $this->loadParticipantes($ids);

        // Ensamblar y calcular
        foreach ($rows as &$r) {
            $pid = (int)$r['id'];
            $lista = $this->mixParticipantes((string)($r['participantes_text'] ?? ''), $parts[$pid] ?? [], (string)($r['chofer_nombre'] ?? ''));

            // días
            $dias = $this->calcDias((string)$r['fecha_inicio'], (string)$r['fecha_fin']);
            $num  = count($lista);

            // viáticos (misma regla del sistema)
            $viaticos = ($dias <= 1) ? (40.0 * $num) : (80.0 * $dias * $num);

            // combustible
            $comb = $this->calcCombustible((string)$r['vehiculo_nombre'], (float)($r['total_km'] ?? 0));

            $r['participantes_num']  = $num;
            $r['dias']               = $dias;
            $r['viaticos_total']     = round($viaticos, 2);
            $r['combustible_costo']  = round((float)$comb['costo_usd'], 2);
            $r['combustible_tipo']   = $comb['tipo'];
            $r['km_por_galon']       = $comb['km_por_galon'];
            $r['precio_galon']       = $comb['precio_galon'];
        } unset($r);

        return $rows;
    }

    /** Top rankings simples */
    public function tops(string $q, string $chofer, string $desde, string $hasta, int $areaId, string $estado): array {
        $rows = $this->comisionesConDatos($q, $chofer, $desde, $hasta, $areaId, $estado);

        // por chofer
        $topChofer = [];
        foreach ($rows as $r) {
            $ch = trim((string)$r['chofer_nombre'] ?: '—');
            if (!isset($topChofer[$ch])) $topChofer[$ch] = ['monto'=>0.0,'count'=>0];
            $topChofer[$ch]['monto'] += (float)$r['viaticos_total'];
            $topChofer[$ch]['count']++;
        }
        uasort($topChofer, fn($a,$b)=> $b['monto'] <=> $a['monto']);
        $topChofer = array_slice($topChofer, 0, 5, true);

        // por comisión (total general)
        $topCom = $rows;
        foreach ($topCom as &$r) $r['gran_total'] = (float)$r['viaticos_total'] + (float)$r['combustible_costo'];
        unset($r);
        usort($topCom, fn($a,$b)=> $b['gran_total'] <=> $a['gran_total']);
        $topCom = array_slice($topCom, 0, 5);

        return ['chofer'=>$topChofer, 'comisiones'=>$topCom];
    }

    /* ============== Helpers internos (copian la lógica del sistema) ============== */

    private function loadParticipantes(array $ids): array {
        if (empty($ids)) return [];
        $db = DB::conn();
        $in = implode(',', array_fill(0, count($ids), '?'));
        $sql = "SELECT cp.comision_id, u.nombre, r.nombre AS rol
                FROM comision_participantes cp
                JOIN users u ON u.id=cp.usuario_id
                LEFT JOIN roles r ON r.id=u.rol_id
                WHERE cp.comision_id IN ($in)";
        $st = $db->prepare($sql);
        $st->execute($ids);
        $out=[];
        while($row = $st->fetch()){
            $out[(int)$row['comision_id']][] = $row;
        }
        return $out;
    }
/** Lista de funcionarios (nombres únicos) presentes en comision_participantes + chips de texto */
public function listarFuncionarios(string $q = ''): array {
    $db = DB::conn();

    // Nombres desde la tabla del sistema
    $sql1 = "SELECT DISTINCT u.nombre AS n FROM comision_participantes cp
             JOIN users u ON u.id=cp.usuario_id
             WHERE u.nombre IS NOT NULL";
    $params1 = [];
    if ($q !== '') { $sql1 .= " AND u.nombre LIKE ?"; $params1[] = "%$q%"; }
    $sql1 .= " ORDER BY u.nombre ASC LIMIT 100";

    $st1 = $db->prepare($sql1);
    $st1->execute($params1);
    $sistema = array_map(fn($r)=>$r['n'], $st1->fetchAll());

    // Nombres desde chips de texto en comisiones.participantes_text (opcional)
    $otros = [];
    $sql2 = "SELECT participantes_text FROM comisiones WHERE participantes_text IS NOT NULL AND participantes_text<>''";
    $st2 = $db->query($sql2);
    while ($r = $st2->fetch()) {
        foreach (array_filter(array_map('trim', explode(',', (string)$r['participantes_text']))) as $n) {
            if ($q!=='' && stripos($n, $q) === false) continue;
            $otros[] = $n;
        }
    }

    // Merge únicos
    $all = array_values(array_unique(array_merge($sistema, $otros)));
    sort($all, SORT_FLAG_CASE | SORT_STRING);
    return array_slice($all, 0, 200);
}

/** Resumen por funcionario (conteo y montos en rango) */
public function resumenFuncionario(string $nombre, string $desde = '', string $hasta = '', string $estado = ''): array {
    $rows = $this->comisionesPorFuncionario($nombre, $desde, $hasta, $estado);
    $totViaticos = 0.0; $totComb = 0.0; $count=0;
    foreach ($rows as $r) {
        $totViaticos += (float)$r['viaticos_total'];
        $totComb     += (float)$r['combustible_costo'];
        $count++;
    }
    return [
        'funcionario'     => $nombre,
        'comisiones'      => $count,
        'viaticos_total'  => round($totViaticos, 2),
        'combustible_total'=> round($totComb, 2),
        'total_general'   => round($totViaticos + $totComb, 2),
    ];
}

/** Detalle de comisiones donde participa el funcionario (sistema o texto) */
public function comisionesPorFuncionario(string $nombre, string $desde = '', string $hasta = '', string $estado = ''): array {
    $db = DB::conn();

    // 1) IDs desde la relación del sistema
    $sqlIds = "SELECT DISTINCT cp.comision_id AS cid
               FROM comision_participantes cp
               JOIN users u ON u.id=cp.usuario_id
               WHERE u.nombre LIKE ?";
    $params = ["%$nombre%"];
    if ($desde !== '') { $sqlIds .= " AND EXISTS(SELECT 1 FROM comisiones c WHERE c.id=cp.comision_id AND c.fecha_inicio >= ?)"; $params[] = $desde; }
    if ($hasta !== '') { $sqlIds .= " AND EXISTS(SELECT 1 FROM comisiones c WHERE c.id=cp.comision_id AND c.fecha_fin <= ?)";    $params[] = $hasta; }
    if ($estado !== ''){ $sqlIds .= " AND EXISTS(SELECT 1 FROM comisiones c WHERE c.id=cp.comision_id AND c.estado = ?)";       $params[] = $estado; }

    $st = $db->prepare($sqlIds); $st->execute($params);
    $idsA = array_map(fn($r)=>(int)$r['cid'], $st->fetchAll());

    // 2) IDs por coincidencia de texto en participantes_text
    $sqlTxt = "SELECT id FROM comisiones
               WHERE participantes_text IS NOT NULL
                 AND participantes_text <> ''
                 AND participantes_text LIKE ?";
    $paramsTxt = ["%$nombre%"];
    if ($desde !== '') { $sqlTxt .= " AND fecha_inicio >= ?"; $paramsTxt[] = $desde; }
    if ($hasta !== '') { $sqlTxt .= " AND fecha_fin <= ?";    $paramsTxt[] = $hasta; }
    if ($estado !== ''){ $sqlTxt .= " AND estado = ?";        $paramsTxt[] = $estado; }

    $st2 = $db->prepare($sqlTxt); $st2->execute($paramsTxt);
    $idsB = array_map(fn($r)=>(int)$r['id'], $st2->fetchAll());

    $ids = array_values(array_unique(array_merge($idsA, $idsB)));
    if (empty($ids)) return [];

    // Traer datos de esas comisiones
    $in = implode(',', array_fill(0, count($ids), '?'));
    $sql = "SELECT c.id, c.titulo, c.descripcion, c.fecha_inicio, c.fecha_fin, c.estado,
                   c.origen_ciudad, c.destino_ida_ciudad, c.destino_retorno_ciudad,
                   c.chofer_nombre, c.vehiculo_nombre, c.total_km,
                   a.nombre AS area_nombre
            FROM comisiones c
            LEFT JOIN areas a ON a.id=c.area_id
            WHERE c.id IN ($in)
            ORDER BY c.fecha_inicio DESC, c.id DESC";
    $st3 = $db->prepare($sql); $st3->execute($ids);
    $rows = $st3->fetchAll();

    // Cálculos — usamos la misma lógica que el sistema
    foreach ($rows as &$r) {
        $dias = $this->calcDias((string)$r['fecha_inicio'], (string)$r['fecha_fin']);
        // viáticos: por persona (1) porque el reporte es por funcionario
        $viaticos = ($dias <= 1) ? 40.0 : (80.0 * $dias);

        $comb = $this->calcCombustible((string)$r['vehiculo_nombre'], (float)($r['total_km'] ?? 0));

        $r['dias']              = $dias;
        $r['viaticos_total']    = round($viaticos, 2);
        $r['combustible_costo'] = round((float)$comb['costo_usd'], 2);
    } unset($r);

    return $rows;
}

    private function mixParticipantes(string $participantesText, array $partSis, string $choferNombre=''): array {
        $list = [];
        foreach (array_filter(array_map('trim', explode(',', $participantesText))) as $c) if ($c!=='') $list[]=$c;
        foreach ($partSis as $ps) { $n = trim($ps['nombre'] ?? ''); if ($n!=='') $list[] = $n; }
        $ch = trim($choferNombre); if ($ch!=='') $list[] = $ch;
        $seen=[]; $final=[];
        foreach ($list as $n){ $k=mb_strtolower($n); if(isset($seen[$k])) continue; $seen[$k]=1; $final[]=$n; }
        return $final;
    }

    private function calcDias(string $ini, string $fin): int {
        try { $d1 = new DateTime($ini); $d2 = new DateTime($fin); return $d1->diff($d2)->days + 1; }
        catch (\Throwable $e){ return 1; }
    }

    private function calcCombustible(?string $vehiculoNombre, ?float $kmTotales): array {
        $vehiculo = strtoupper(trim((string)$vehiculoNombre));
        $tabla = [
            'TOYOTA FORTUNES'      => ['tipo'=>'GASOLINA','km_gal'=>22.0],
            'TOYOTA HILUX'         => ['tipo'=>'GASOLINA','km_gal'=>36.0],
            'CHEVROLET LUV DIMAX'  => ['tipo'=>'DIÉSEL',  'km_gal'=>38.0],
            'VITARA 3 PUERTAS'     => ['tipo'=>'GASOLINA','km_gal'=>50.0],
        ];
        $kmPorGal = $tabla[$vehiculo]['km_gal'] ?? 30.0;
        $tipo     = $tabla[$vehiculo]['tipo']   ?? 'GASOLINA';
        $km = round(max(0.0, (float)$kmTotales), 2);

        $precioGas    = defined('FUEL_GAS_PRICE_USD')    ? (float)FUEL_GAS_PRICE_USD    : 2.88;
        $precioDiesel = defined('FUEL_DIESEL_PRICE_USD') ? (float)FUEL_DIESEL_PRICE_USD : 2.80;
        $precioGal = ($tipo === 'DIÉSEL') ? $precioDiesel : $precioGas;

        $galones = $km > 0 ? ($km / $kmPorGal) : 0.0;
        $costo   = $galones * $precioGal;

        return [
            'tipo'=>$tipo,
            'km_por_galon'=>$kmPorGal,
            'precio_galon'=>$precioGal,
            'costo_usd'=>$costo
        ];
    }
}
